# Get-AD-Export-Script-v3.0.0.ps1
#
param (
    [string]
    $domainControllerName,

    [string]
    $domainServerPort,

    [string]
    $domainAccountUsername,

    [string]
    $domainAccountPassword,

    [string]
    $domainAccountServer,

    [string]
    $file = '.\ADUsers.ldif',

    [string]
    $uploadUsername,

    [string]
    $uploadPassword,

    [switch]
    $CreatePasswordFile = $false,

    [string]
    $passwordFile,

    [string]
    $auxAttibutesList
)

# The following script uploads a file to a customer's folder in a GCP Cloud Storage Bucket
# The username and password are that of a Customer representative's Filemage account.

function Upload-File() {
param(
        [string]
        $uploadUsername,
    
        [string]
        $uploadPassword,
    
        [string]
        $file
    )
    
    $filemageSigninUrl = 'https://upload.access-analytics.com/signin/'
    $filemageGetUploadUrl = 'https://upload.access-analytics.com/files/upload/'
    
    # Prepare the error object to return
    $result = @{
        error         = $false
        errorMessages = New-Object 'System.Collections.Generic.List[string]'
    }
    
    # Set TLS 1.2 as default protocall for Azure AD connections
    $TLS12Protocol = [System.Net.SecurityProtocolType] 'Ssl3 , Tls12'
    [System.Net.ServicePointManager]::SecurityProtocol = $TLS12Protocol
    
    # Attempt to authenticate and obtain a 'w' token
    try {
        $uploadFilename = Split-Path $file -leaf
        Write-Host $uploadFilename
        Write-Host $file
        $body = @{
            username = $uploadUsername
            password = $uploadPassword
        }
        $requestParams = @{
            Method = 'POST'
            Uri = $filemageSigninUrl
            SessionVariable = 'session'
            Body = $body
        }
        # Invoke and return a session
        $response = Invoke-WebRequest @requestParams -UseBasicParsing
    } catch {
        $result.error = $true
        $result.StatusCode = $response.StatusCode
        $result.StatusDescription = $response.StatusDescription
        $result.errorMessages = @("Could not authenticate. Check username or password.")
        return $result
    }
    
    # Attempt to upload the file
    try {
        # Use the session created in the (previous) authN request and pass the 'w' Set-Cookie. Obtain the GCP Bucket URL.
        $body = @{
            contentType = 'text/plain'
            path = $uploadFilename
        }
        $requestParams = @{
            Method = 'POST'
            Uri = $filemageGetUploadUrl
            ContentType = 'application/json'
            Body = ($body | ConvertTo-Json -compress)
            WebSession = $session
        }
        # Invoke and return a URL in the payload for where to PUT the file
        $response = Invoke-WebRequest @requestParams -UseBasicParsing
        $uploadUrl = ($response.Content | ConvertFrom-Json).url
    
        # only proceed if a seemingly valid URL is returned
        if(([string]::IsNullOrWhiteSpace($uploadUrl)) -or $response.StatusCode -ne 200) {
            $result.error = $true
            $result.StatusCode = $response.StatusCode
            $result.StatusDescription = $response.StatusDescription
            $result.files = $body.path
            $result.errorMessages = @("Could not prepare for upload.")
            return $result
        }
    
        # PUT the file to the URL provided in the previous request's response
        $requestParams = @{
            Method = 'PUT'
            Uri = $uploadUrl
            ContentType = 'text/plain'
            Infile = $file
        }
        # Invoke the PUT request
        $response = Invoke-WebRequest @requestParams -UseBasicParsing -DisableKeepAlive
        Write-Host $response.StatusCode
        Remove-Variable -Name 'session' -Scope global -ErrorAction SilentlyContinue

        if ($response.StatusCode -eq 200) {
            # Successful upload
            $result.error = $false
            $result.files = $body.path
            $result.errorMessages = @("File upload sucessful.")
            return $result
        } else {
            # Something went wrong
            $result.error = $true
            $result.StatusCode = $response.StatusCode
            $result.StatusDescription = $response.StatusDescription
            $result.files = $body.path
            $result.errorMessages = @("File upload failed.")
            return $result
        }
    } catch {
        # Unknown issue
        $result.error = $true
        $result.files = $body.path
        $result.errorMessages = @("Could not upload file.")
        return $result
    }
    
}

# Create Password File if requested
if ($CreatePasswordFile) {
    try {
        $databasePasswordSec = Read-Host -Prompt "Upload password" -AsSecureString | ConvertFrom-SecureString | Out-File $passwordFile
        Write-Host "Encrypted password is stored in $passwordFile"
    }
    catch {
        Write-Host "The passwordFile argument is not provided" -ForegroundColor Red
    }
    exit 0
}

# check is empty or null uplaod credetial and exit before export
if ($uploadUsername) {
    if ($uploadPassword) {
        if ($passwordFile) {
            Write-Host "Both uploadPassword or passwordFile are provided. Please provide only one" -ForegroundColor Red
            Exit 1
        }
        # all good for upload, proceed with extract
    }
    else {
        if ($passwordFile) {
            # Decript password from password file
            if ($null -ne $passwordFile -and $passwordFile -ne '' -and (Test-Path -Path $passwordFile)) {
                $len = (Get-Item $passwordFile).length 
                if ($len -lt 5) {
                    Write-Host "Password file empty or to short to contain encryted key" -ForegroundColor Red
                    Exit 1   
                } 
                $uploadPassword = ''
                try {
                    $uploadPasswordEnc = Get-Content $passwordFile
                    $uploadPasswordSec = ConvertTo-SecureString $uploadPasswordEnc
                    $uploadPassword = [Runtime.InteropServices.Marshal]::PtrToStringAuto([Runtime.InteropServices.Marshal]::SecureStringToBSTR($uploadPasswordSec))
                }
                catch {
                    Write-Host "Failed to decrypt password from $passwordFile" -ForegroundColor Red
                    Exit 1
                }
            }
        }
        else {
            Write-Host "Missing uploadPassword or passwordFile" -ForegroundColor Red
            Exit 1    
        }
    }
}

# Put this in a try catch, mainly so we can execute the finally, which will always run
try {

    # List of block attributes
    $b1 = "accountExpires,badPasswordTime,badPwdCount,c,cn,co,company,countryCode,department,description,directReports,displayName,distinguishedName,dn,employeeID,givenName,groupType,l,"
    $b2 = "lastLogon,lastLogonTimestamp,lockoutTime,logonCount,mail,manager,memberOf,name,objectCategory,objectClass,objectGUID,objectSid,primaryGroupID,pwdLastSet,sAMAccountName,"
    $b3 = "sAMAccountType,sn,st,status,telephoneNumber,title,userAccountControl,userAccountControlFlags,userPrincipalName,whenChanged,whenCreated,wWWHomePage"
    $whitelistAttrs = $b1 + $b2 + $b3

    if ($auxAttibutesList) {
        $whitelistAttrs = $whitelistAttrs + ',' + $auxAttibutesList
    }

    # Start developing the arguments
    $ldifdeArguments = @("-r", "`"(|(objectClass=user)(objectClass=computer)(objectClass=domain)(objectClass=group)(objectClass=foreignSecurityPrincipal))`"", '-l', $whitelistAttrs)


    # Do we have a domain?
    if ([string]::IsNullOrWhiteSpace($domainControllerName)) { 
        Write-Host "The 'LDIFDE.exe' settings did not include a server to perform the data extraction on"; exit 1 
    }

    # Include the domain
    else { $ldifdeArguments += @('-s', $domainControllerName) }


    # Is the port a number?
    if ([regex]::IsMatch($domainServerPort, '^\d{1,5}$', 1)) { $ldifdeArguments += @('-t', [int]$domainServerPort) }

    # Otherwise, use the secure port
    else {  Write-Host "No valid port was provided for 'LDIFDE.exe', using '636'"; $ldifdeArguments += @('-t', 636) }


    # Is the provided path completely invalid?
    if ([string]::IsNullOrWhiteSpace($file)) {  Write-Host "The file path provided to 'LDIFDE.exe' was blank"; exit 1 }

    # Is this not a valid directory?
    elseif (-not (Test-Path -LiteralPath $file -IsValid)) {  Write-Host  "The file path of '$file' provided to 'LDIFDE.exe' was not a valid file path"; exit 1 }

    # Otherwise, the file is fine
    else { $ldifdeArguments += @('-f', $file) }


    # We have a username, but not a password
    if ($domainAccountUsername -and -not $domainAccountPassword) { Write-Host  "The 'LDIFDE.exe' settings included a username, but did not include a password"; exit 1 }

    # We have a password, but not a username
    elseif (-not $domainAccountUsername -and $domainAccountPassword) { Write-Host 'Settings included a password, but did not include a username'; exit 1  }

    # We have a password and a username, but no domain
    elseif ($domainAccountUsername -and $domainAccountPassword -and -not $domainAccountServer) { $ldifdeArguments += @('-a', $domainAccountUsername, $domainAccountPassword) }

    # We have a password and a username, but no domain
    elseif ($domainAccountUsername -and $domainAccountPassword -and $domainAccountServer) { $ldifdeArguments += @('-b', $domainAccountUsername, $domainAccountServer, $domainAccountPassword) }
}

# Catch and log the error
catch { 
#    Write-Host "An Exception was encountered while attempting to develop the LDIFDE.exe tool arguments, Exception type: '$($_.Exception.GetType().fullname)'"; 
    # Assign the error object to return
    $result = @{
        error         = $true
        errorMessages = "An Exception was encountered while attempting to develop the LDIFDE.exe tool arguments, Exception type: '$($_.Exception.GetType().fullname)'"
        files         = @($destination)
    }
    Write-Host "Uploaded files $($result.errorMessages)"

    # Return the error
    return $result

}

# Always execute this code, which removes the account password from all scopes
finally {

    # Clear the 'ldifSplatParams' variable
    Clear-Variable -Name 'domainAccountPassword' -Scope local  -ErrorAction SilentlyContinue
    Clear-Variable -Name 'domainAccountPassword' -Scope script -ErrorAction SilentlyContinue
    Clear-Variable -Name 'domainAccountPassword' -Scope global -ErrorAction SilentlyContinue

    # Remove the 'ldifSplatParams' variable
    Remove-Variable -Name 'domainAccountPassword' -Scope local  -ErrorAction SilentlyContinue
    Remove-Variable -Name 'domainAccountPassword' -Scope script -ErrorAction SilentlyContinue
    Remove-Variable -Name 'domainAccountPassword' -Scope global -ErrorAction SilentlyContinue
}

# Run LDIFDE with ldifdeArguments
ldifde.exe $ldifdeArguments
$result = @{
    error         = $false
    errorMessages = "LDIFDE completed"
    files         = @($file)
}

# check is empty or null
if ($uploadUsername) {
    if ($uploadPassword) {
        # upload file
        $ProgressPreference = 'SilentlyContinue'
        $result = Upload-File $uploadUsername $uploadPassword $file
        if ($result.error) {
           Write-Host ($result | Format-List | Out-String)
#           Exit 1
        } else {
            Write-Host "Uploaded $file"
        # remove file
           Remove-Item -Path $file
#           Exit 0
        }
    }
}
return $result
